<?php
session_start();
include 'includes/auth_check.php';
include 'includes/config.php';

require __DIR__ . '/vendor/autoload.php';

use setasign\Fpdi\Tcpdf\Fpdi;
use chillerlan\QRCode\{QRCode, QROptions};

$uploadDir = __DIR__ . '/uploads/';
$outputDir = __DIR__ . '/processed/';
$logoPath = __DIR__ . '/assets/images/logo.png';
$baseUrl = 'http://localhost/laboratorio/processed/'; // URL base ajustada

if (!is_dir($uploadDir) && !mkdir($uploadDir, 0755, true)) {
    die('Error creando carpeta uploads');
}
if (!is_dir($outputDir) && !mkdir($outputDir, 0755, true)) {
    die('Error creando carpeta processed');
}

$usuarioId = $_SESSION['usuario_id'] ?? null;
if (!$usuarioId) {
    die("No has iniciado sesión.");
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST' || !isset($_FILES['pdfFile'])) {
    die('No se recibió archivo PDF.');
}

$file = $_FILES['pdfFile'];
if ($file['error'] !== UPLOAD_ERR_OK) {
    die('Error en la carga del archivo: ' . $file['error']);
}

$ext = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
if ($ext !== 'pdf') {
    die('Solo se permiten archivos PDF.');
}

$uploadedName = uniqid('upl_') . '.' . $ext;
$uploadedPath = $uploadDir . $uploadedName;
if (!move_uploaded_file($file['tmp_name'], $uploadedPath)) {
    die('No se pudo guardar el archivo.');
}

try {
    $claveSecreta = str_pad(random_int(0, 999999), 6, '0', STR_PAD_LEFT);
} catch (Exception $e) {
    unlink($uploadedPath);
    die('Error generando clave secreta: ' . $e->getMessage());
}

$pdfMod = new Fpdi();
$pageCount = $pdfMod->setSourceFile($uploadedPath);
if ($pageCount < 1) {
    unlink($uploadedPath);
    die('PDF inválido');
}

$outputFileName = 'documento_' . uniqid() . '.pdf';
$outputFilePath = $outputDir . $outputFileName;
$fullUrl = $baseUrl . basename($outputFilePath); 

$options = new QROptions([
    'eccLevel' => QRCode::ECC_H,
    'outputType' => QRCode::OUTPUT_IMAGE_PNG,
    'scale' => 10,
    'imageBase64' => false,
    'addQuietzone' => true,
]);

$qr = new QRCode($options);
$qrImageString = $qr->render($fullUrl);

if (!file_exists($logoPath)) {
    unlink($uploadedPath);
    die("Logo no encontrado en: $logoPath");
}

$qrImage = imagecreatefromstring($qrImageString);
$logo = createHighQualityLogo($logoPath);

$finalWidth = 300; 
$finalHeight = 300; 
$finalImage = imagecreatetruecolor($finalWidth, $finalHeight);
$white = imagecolorallocate($finalImage, 255, 255, 255);
imagefill($finalImage, 0, 0, $white);
imagecopyresampled($finalImage, $qrImage, 0, 0, 0, 0, $finalWidth, $finalHeight, imagesx($qrImage), imagesy($qrImage));

list($logoW, $logoH) = calculateLogoSize($logo, $finalWidth);
$logoX = ($finalWidth - $logoW) / 2;
$logoY = ($finalHeight - $logoH) / 2;
imagecopyresampled($finalImage, $logo, $logoX, $logoY, 0, 0, $logoW, $logoH, imagesx($logo), imagesy($logo));

$qrFileName = 'qr_' . uniqid() . '.png';
$qrFilePath = $uploadDir . $qrFileName;
imagepng($finalImage, $qrFilePath, 9);
imagedestroy($finalImage);
imagedestroy($logo);

for ($i = 1; $i <= $pageCount; $i++) {
    $pageId = $pdfMod->importPage($i);
    $size = $pdfMod->getTemplateSize($pageId);
    
    $pdfMod->AddPage($size['orientation'], [$size['width'], $size['height']]);
    $pdfMod->useTemplate($pageId);
    
    if ($i === 1) {
        $qrX = 158.1;
        $qrY = $size['height'] - 258.8; 
        $pdfMod->Image($qrFilePath, $qrX, $qrY, 50, 50, 'PNG', '', 'N', false, 300, '', false, false, 0, false, false, true);
    }
}

$pdfMod->Output($outputFilePath, 'F');
unlink($uploadedPath);

try {
    $pdo = new PDO("mysql:host=localhost;dbname=laboratorio;charset=utf8mb4", "root", "", [
        PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
    ]);
    $stmt = $pdo->prepare("INSERT INTO archivos_pdf (nombre_pdf, url_pdf, clave_secreta, fecha_subida, usuario_subida_id) VALUES
        (:nombre_pdf, :url_pdf, :clave_secreta, NOW(), :usuario_subida_id)");
    $stmt->execute([
        ':nombre_pdf' => $file['name'],
        ':url_pdf' => $fullUrl, 
        ':clave_secreta' => $claveSecreta,
        ':usuario_subida_id' => $usuarioId
    ]);
} catch (PDOException $e) {
    unlink($qrFilePath);
    unlink($outputFilePath);
    die("Error de base de datos: " . $e->getMessage());
}

echo json_encode(['success' => true, 'url' => $fullUrl]);

function createHighQualityLogo($path) {
    $logo = imagecreatefrompng($path);
    $width = imagesx($logo);
    $height = imagesy($logo);
    
    $highQuality = imagecreatetruecolor($width, $height);
    imagealphablending($highQuality, false);
    imagesavealpha($highQuality, true);
    imagecopyresampled($highQuality, $logo, 0, 0, 0, 0, $width, $height, $width, $height);
    
    return $highQuality;
}

function calculateLogoSize($logo, $qrSize) {
    $maxLogoSize = $qrSize * 0.4;
    $logoW = imagesx($logo);
    $logoH = imagesy($logo);
    
    $ratio = $logoW / $logoH;
    if ($logoW > $logoH) {
        $newW = min($maxLogoSize, $logoW);
        $newH = $newW / $ratio;
    } else {
        $newH = min($maxLogoSize, $logoH);
        $newW = $newH * $ratio;
    }
    
    return [round($newW), round($newH)];
}
?>
